# Arquitetura da Aplicação para Análise de Vídeos do Facebook

## Visão Geral da Arquitetura

A aplicação será desenvolvida utilizando o framework Laravel (PHP) com banco de dados MySQL e interface de usuário construída com Bootstrap. A arquitetura seguirá o padrão MVC (Model-View-Controller) do Laravel, com adição de camadas de serviço e repositório para melhor separação de responsabilidades.

## Estrutura de Diretórios

```
facebook-video-analytics/
├── app/
│   ├── Console/
│   │   └── Commands/                  # Comandos personalizados para tarefas agendadas
│   ├── Exceptions/                    # Tratamento de exceções personalizadas
│   ├── Http/
│   │   ├── Controllers/               # Controllers da aplicação
│   │   │   ├── Auth/                  # Controllers de autenticação
│   │   │   ├── Dashboard/             # Controllers do dashboard
│   │   │   ├── Video/                 # Controllers relacionados a vídeos
│   │   │   └── Api/                   # Controllers da API interna
│   │   ├── Middleware/                # Middlewares personalizados
│   │   └── Requests/                  # Form requests para validação
│   ├── Jobs/                          # Jobs para processamento em fila
│   │   ├── FetchVideoInsights.php     # Job para buscar insights de vídeos
│   │   ├── ProcessVideoMetrics.php    # Job para processar métricas de vídeos
│   │   └── GenerateReports.php        # Job para gerar relatórios
│   ├── Models/                        # Modelos Eloquent
│   │   ├── User.php
│   │   ├── Page.php
│   │   ├── Video.php
│   │   ├── VideoMetric.php
│   │   ├── AudienceSegment.php
│   │   └── Report.php
│   ├── Providers/                     # Service providers
│   ├── Repositories/                  # Camada de repositório para acesso a dados
│   │   ├── Interfaces/
│   │   └── Eloquent/
│   └── Services/                      # Camada de serviço para lógica de negócios
│       ├── Facebook/                  # Serviços relacionados à API do Facebook
│       │   ├── FacebookApiService.php
│       │   ├── VideoInsightService.php
│       │   └── AuthenticationService.php
│       ├── Analytics/                 # Serviços de análise de dados
│       │   ├── MetricAnalysisService.php
│       │   ├── AudienceAnalysisService.php
│       │   └── TrendAnalysisService.php
│       └── Report/                    # Serviços de relatórios
│           └── ReportGenerationService.php
├── bootstrap/                         # Arquivos de inicialização do Laravel
├── config/                            # Configurações da aplicação
│   ├── facebook.php                   # Configurações específicas do Facebook
│   └── analytics.php                  # Configurações de análise
├── database/
│   ├── factories/                     # Factories para testes
│   ├── migrations/                    # Migrações do banco de dados
│   └── seeders/                       # Seeders para dados iniciais
├── public/                            # Arquivos públicos
│   ├── css/
│   ├── js/
│   └── images/
├── resources/
│   ├── js/                            # Arquivos JavaScript
│   │   └── components/                # Componentes Vue.js (opcional)
│   ├── sass/                          # Arquivos SASS
│   └── views/                         # Templates Blade
│       ├── auth/
│       ├── dashboard/
│       ├── videos/
│       ├── reports/
│       ├── layouts/
│       └── components/
├── routes/
│   ├── web.php                        # Rotas web
│   ├── api.php                        # Rotas API
│   └── channels.php                   # Canais de broadcast
├── storage/                           # Armazenamento de arquivos
├── tests/                             # Testes automatizados
└── vendor/                            # Dependências (gerenciadas pelo Composer)
```

## Estrutura do Banco de Dados

### Tabelas Principais

1. **users**
   - id (PK)
   - name
   - email
   - password
   - remember_token
   - facebook_token
   - facebook_token_expires_at
   - created_at
   - updated_at

2. **pages**
   - id (PK)
   - user_id (FK)
   - facebook_page_id
   - name
   - access_token
   - token_expires_at
   - category
   - follower_count
   - created_at
   - updated_at

3. **videos**
   - id (PK)
   - page_id (FK)
   - facebook_video_id
   - title
   - description
   - thumbnail_url
   - permalink_url
   - duration
   - published_at
   - video_type (regular, reel)
   - status
   - created_at
   - updated_at

4. **video_metrics**
   - id (PK)
   - video_id (FK)
   - metric_name
   - metric_value
   - period
   - date_start
   - date_end
   - created_at
   - updated_at

5. **audience_segments**
   - id (PK)
   - video_id (FK)
   - segment_type (age, gender, location, etc.)
   - segment_value
   - view_count
   - engagement_rate
   - created_at
   - updated_at

6. **topics**
   - id (PK)
   - name
   - description
   - created_at
   - updated_at

7. **video_topics**
   - id (PK)
   - video_id (FK)
   - topic_id (FK)
   - relevance_score
   - created_at
   - updated_at

8. **reports**
   - id (PK)
   - user_id (FK)
   - title
   - description
   - type
   - parameters (JSON)
   - created_at
   - updated_at

9. **report_results**
   - id (PK)
   - report_id (FK)
   - result_data (JSON)
   - generated_at
   - created_at
   - updated_at

### Migrações

As migrações serão criadas seguindo a convenção do Laravel:

```php
// Exemplo de migração para a tabela videos
public function up()
{
    Schema::create('videos', function (Blueprint $table) {
        $table->id();
        $table->foreignId('page_id')->constrained()->onDelete('cascade');
        $table->string('facebook_video_id')->unique();
        $table->string('title')->nullable();
        $table->text('description')->nullable();
        $table->string('thumbnail_url')->nullable();
        $table->string('permalink_url')->nullable();
        $table->integer('duration')->nullable();
        $table->dateTime('published_at');
        $table->enum('video_type', ['regular', 'reel'])->default('regular');
        $table->string('status')->default('active');
        $table->timestamps();
        
        $table->index('facebook_video_id');
        $table->index('published_at');
    });
}
```

## Componentes Principais

### 1. Módulo de Autenticação e Autorização

- Autenticação de usuários com Laravel Fortify/Breeze
- Integração com Facebook Login
- Gerenciamento de tokens de acesso do Facebook
- Renovação automática de tokens expirados
- Controle de permissões baseado em funções

### 2. Módulo de Integração com Facebook

- Serviço de autenticação com Facebook
- Serviço de busca de páginas do usuário
- Serviço de busca de vídeos por página
- Serviço de busca de métricas de vídeos
- Serviço de busca de dados demográficos

### 3. Módulo de Análise de Dados

- Processamento e normalização de métricas
- Análise de tendências por tema
- Segmentação de público
- Classificação de vídeos por desempenho
- Geração de insights baseados em dados

### 4. Módulo de Dashboard e Visualização

- Dashboard principal com KPIs
- Gráficos de desempenho de vídeos
- Visualização de dados demográficos
- Comparação de vídeos por tema
- Exportação de relatórios

### 5. Módulo de Agendamento e Processamento em Segundo Plano

- Jobs para busca periódica de dados
- Filas para processamento assíncrono
- Notificações de relatórios concluídos
- Cache de dados frequentemente acessados

## Fluxos Principais

### 1. Fluxo de Autenticação

1. Usuário acessa a aplicação
2. Usuário realiza login com credenciais ou via Facebook
3. Sistema solicita permissões necessárias do Facebook
4. Sistema armazena tokens de acesso
5. Usuário é redirecionado para o dashboard

### 2. Fluxo de Importação de Vídeos

1. Usuário seleciona página do Facebook
2. Sistema busca vídeos disponíveis na página
3. Sistema armazena metadados dos vídeos
4. Sistema agenda job para buscar métricas de cada vídeo
5. Usuário visualiza lista de vídeos importados

### 3. Fluxo de Análise por Tema

1. Usuário define tema para análise
2. Sistema filtra vídeos relacionados ao tema
3. Sistema processa métricas dos vídeos filtrados
4. Sistema identifica vídeos com melhor desempenho
5. Sistema analisa características do público engajado
6. Usuário visualiza relatório de análise

### 4. Fluxo de Geração de Relatórios

1. Usuário configura parâmetros do relatório
2. Sistema agenda job para geração do relatório
3. Sistema processa dados e gera visualizações
4. Sistema notifica usuário quando relatório estiver pronto
5. Usuário visualiza ou exporta relatório

## Camadas da Aplicação

### 1. Camada de Apresentação (Views e Controllers)

- **Controllers**: Responsáveis por receber requisições, validar entradas e coordenar respostas
- **Views**: Templates Blade com Bootstrap para interface do usuário
- **Form Requests**: Validação de entradas do usuário

### 2. Camada de Serviço

- Implementação da lógica de negócios
- Orquestração de chamadas entre repositórios
- Processamento e transformação de dados
- Integração com APIs externas

### 3. Camada de Repositório

- Abstração do acesso a dados
- Implementação de consultas complexas
- Cache de resultados frequentes
- Transações e integridade de dados

### 4. Camada de Modelo

- Modelos Eloquent representando entidades do sistema
- Relacionamentos entre entidades
- Mutators e Accessors para transformação de dados
- Scopes para consultas comuns

## Considerações Técnicas

### 1. Performance

- Uso de jobs em segundo plano para operações demoradas
- Implementação de cache para dados frequentemente acessados
- Paginação de resultados grandes
- Índices adequados no banco de dados
- Eager loading para evitar problema N+1

### 2. Segurança

- Armazenamento seguro de tokens
- Validação de todas as entradas do usuário
- Proteção contra CSRF
- Autenticação e autorização em todas as rotas
- Sanitização de saídas

### 3. Escalabilidade

- Separação clara de responsabilidades
- Uso de filas para processamento assíncrono
- Modularização do código
- Limitação de tamanho de classes (máximo 450 linhas)
- Testes automatizados

### 4. Manutenibilidade

- Documentação de código
- Padrões de codificação consistentes
- Injeção de dependência
- Uso de interfaces para desacoplamento
- Logs detalhados

## Tecnologias e Bibliotecas

### Backend

- **Laravel**: Framework PHP para desenvolvimento web
- **MySQL**: Sistema de gerenciamento de banco de dados
- **Laravel Horizon**: Dashboard para gerenciamento de filas
- **Laravel Socialite**: Integração com OAuth para Facebook
- **Guzzle**: Cliente HTTP para requisições à API do Facebook
- **Carbon**: Manipulação de datas e horários
- **Spatie Permission**: Gerenciamento de permissões e funções

### Frontend

- **Bootstrap**: Framework CSS para interface responsiva
- **Chart.js**: Biblioteca para criação de gráficos
- **DataTables**: Plugin jQuery para tabelas interativas
- **Select2**: Plugin jQuery para campos select avançados
- **SweetAlert2**: Biblioteca para alertas e modais personalizados
- **Font Awesome**: Biblioteca de ícones

## Considerações de Implantação

- Servidor web com suporte a PHP 8.1+
- Servidor MySQL 8.0+
- Configuração de filas (recomendado Redis)
- Configuração de tarefas agendadas (cron)
- Certificado SSL para HTTPS
- Variáveis de ambiente para configurações sensíveis
- Backup regular do banco de dados
