<?php

namespace App\Http\Controllers;

use App\Models\VideoAd;
use App\Services\AdLibraryService;
use Illuminate\Http\Request;

class VideoAdController extends Controller
{
    /**
     * Serviço de Ad Library
     *
     * @var AdLibraryService
     */
    protected $adLibraryService;

    /**
     * Construtor
     *
     * @param AdLibraryService $adLibraryService
     */
    public function __construct(AdLibraryService $adLibraryService)
    {
        $this->adLibraryService = $adLibraryService;
    }

    /**
     * Exibe a página inicial com a interface de pesquisa
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        // Obtém as categorias disponíveis para o filtro
        $categories = [
            'Moda',
            'Eletrônicos',
            'Casa e Decoração',
            'Alimentos e Bebidas',
            'Beleza e Cuidados Pessoais',
            'Esportes',
            'Brinquedos e Jogos',
            'Livros e Mídia',
            'Saúde e Bem-estar',
            'Outros'
        ];

        return view('video_ads.index', compact('categories'));
    }

    /**
     * Busca anúncios de vídeo com base nos filtros
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function search(Request $request)
    {
        $searchTerm = $request->input('search_term', '');
        $category = $request->input('category', '');
        $minViews = $request->input('min_views', 0);
        $minShares = $request->input('min_shares', 0);
        $minConversions = $request->input('min_conversions', 0);

        // Busca anúncios na API ou no banco de dados local
        $searchParams = [
            'search_terms' => $searchTerm,
        ];

        if (!empty($category)) {
            $searchParams['category'] = $category;
        }

        // Busca na API do Facebook
        $results = $this->adLibraryService->searchRetailVideoAds($searchParams);

        // Filtra os resultados com base nos critérios adicionais
        $filteredResults = collect($results['data'])->filter(function ($ad) use ($minViews, $minShares, $minConversions) {
            return $ad['views'] >= $minViews &&
                   $ad['shares'] >= $minShares &&
                   $ad['conversions'] >= $minConversions;
        })->values()->all();

        // Obtém as categorias disponíveis para o filtro
        $categories = [
            'Moda',
            'Eletrônicos',
            'Casa e Decoração',
            'Alimentos e Bebidas',
            'Beleza e Cuidados Pessoais',
            'Esportes',
            'Brinquedos e Jogos',
            'Livros e Mídia',
            'Saúde e Bem-estar',
            'Outros'
        ];

        return view('video_ads.search_results', [
            'ads' => $filteredResults,
            'categories' => $categories,
            'searchTerm' => $searchTerm,
            'category' => $category,
            'minViews' => $minViews,
            'minShares' => $minShares,
            'minConversions' => $minConversions,
            'totalResults' => count($filteredResults)
        ]);
    }

    /**
     * Exibe detalhes de um anúncio específico
     *
     * @param string $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        // Busca detalhes do anúncio na API
        $ad = $this->adLibraryService->getAdDetails($id);

        if (isset($ad['error'])) {
            return redirect()->route('video_ads.index')
                ->with('error', 'Anúncio não encontrado.');
        }

        return view('video_ads.show', ['ad' => $ad]);
    }

    /**
     * Exibe análise comparativa de anúncios
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function compare(Request $request)
    {
        $adIds = $request->input('ads', []);

        if (empty($adIds) || count($adIds) < 2) {
            return redirect()->route('video_ads.index')
                ->with('error', 'Selecione pelo menos dois anúncios para comparar.');
        }

        $ads = [];
        foreach ($adIds as $id) {
            $ads[] = $this->adLibraryService->getAdDetails($id);
        }

        return view('video_ads.compare', ['ads' => $ads]);
    }

    /**
     * Exibe análise de público-alvo para um anúncio
     *
     * @param string $id
     * @return \Illuminate\View\View
     */
    public function audience($id)
    {
        // Busca detalhes do anúncio na API
        $ad = $this->adLibraryService->getAdDetails($id);

        if (isset($ad['error'])) {
            return redirect()->route('video_ads.index')
                ->with('error', 'Anúncio não encontrado.');
        }

        return view('video_ads.audience', ['ad' => $ad]);
    }

    /**
     * Exibe análise de desempenho para um anúncio
     *
     * @param string $id
     * @return \Illuminate\View\View
     */
    public function performance($id)
    {
        // Busca detalhes do anúncio na API
        $ad = $this->adLibraryService->getAdDetails($id);

        if (isset($ad['error'])) {
            return redirect()->route('video_ads.index')
                ->with('error', 'Anúncio não encontrado.');
        }

        return view('video_ads.performance', ['ad' => $ad]);
    }
}
