<?php

namespace App\Services;

use App\Models\Topic;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

/**
 * Serviço para categorização de vídeos por temas
 */
class TopicCategorizer
{
    /**
     * Categorias principais do segmento de varejo
     *
     * @var array
     */
    protected $retailCategories = [
        'Moda',
        'Eletrônicos',
        'Casa e Decoração',
        'Alimentos e Bebidas',
        'Beleza e Cuidados Pessoais',
        'Esportes',
        'Brinquedos e Jogos',
        'Livros e Mídia',
        'Saúde e Bem-estar',
        'Outros'
    ];

    /**
     * Subcategorias por categoria principal
     *
     * @var array
     */
    protected $subCategories = [
        'Moda' => [
            'Roupas Femininas',
            'Roupas Masculinas',
            'Calçados',
            'Acessórios',
            'Moda Infantil',
            'Moda Praia',
            'Moda Fitness',
            'Moda Plus Size',
            'Moda Sustentável',
            'Outros'
        ],
        'Eletrônicos' => [
            'Smartphones',
            'Computadores',
            'Tablets',
            'TVs',
            'Áudio',
            'Câmeras',
            'Videogames',
            'Acessórios Eletrônicos',
            'Eletrodomésticos',
            'Outros'
        ],
        'Casa e Decoração' => [
            'Móveis',
            'Decoração',
            'Utensílios Domésticos',
            'Cama, Mesa e Banho',
            'Iluminação',
            'Jardim',
            'Organização',
            'Eletrodomésticos',
            'Reforma e Construção',
            'Outros'
        ],
        'Alimentos e Bebidas' => [
            'Alimentos Básicos',
            'Bebidas',
            'Alimentos Saudáveis',
            'Doces e Sobremesas',
            'Produtos Gourmet',
            'Bebidas Alcoólicas',
            'Suplementos Alimentares',
            'Alimentos Orgânicos',
            'Alimentos Congelados',
            'Outros'
        ],
        'Beleza e Cuidados Pessoais' => [
            'Maquiagem',
            'Cuidados com a Pele',
            'Cuidados com o Cabelo',
            'Perfumes',
            'Higiene Pessoal',
            'Produtos Naturais',
            'Cuidados Masculinos',
            'Manicure e Pedicure',
            'Equipamentos de Beleza',
            'Outros'
        ],
        'Esportes' => [
            'Roupas Esportivas',
            'Calçados Esportivos',
            'Equipamentos Esportivos',
            'Suplementos',
            'Acessórios Esportivos',
            'Esportes Aquáticos',
            'Esportes Coletivos',
            'Esportes Individuais',
            'Esportes Radicais',
            'Outros'
        ],
        'Brinquedos e Jogos' => [
            'Brinquedos Educativos',
            'Jogos de Tabuleiro',
            'Bonecos e Figuras de Ação',
            'Brinquedos Eletrônicos',
            'Brinquedos para Bebês',
            'Jogos de Cartas',
            'Brinquedos de Montar',
            'Brinquedos ao Ar Livre',
            'Pelúcias',
            'Outros'
        ],
        'Livros e Mídia' => [
            'Livros Físicos',
            'E-books',
            'Audiolivros',
            'Filmes e Séries',
            'Música',
            'Revistas',
            'Quadrinhos',
            'Material Didático',
            'Papelaria',
            'Outros'
        ],
        'Saúde e Bem-estar' => [
            'Medicamentos',
            'Vitaminas e Suplementos',
            'Equipamentos Médicos',
            'Produtos Naturais',
            'Cuidados com Idosos',
            'Ortopedia',
            'Saúde Sexual',
            'Primeiros Socorros',
            'Fitness e Bem-estar',
            'Outros'
        ],
        'Outros' => [
            'Pet Shop',
            'Automotivo',
            'Instrumentos Musicais',
            'Artigos Religiosos',
            'Artesanato',
            'Produtos Infantis',
            'Produtos Corporativos',
            'Produtos Personalizados',
            'Serviços',
            'Diversos'
        ]
    ];

    /**
     * Palavras-chave associadas a cada categoria
     *
     * @var array
     */
    protected $categoryKeywords = [
        'Moda' => [
            'roupa', 'vestido', 'calça', 'camisa', 'sapato', 'tênis', 'bolsa', 'acessório', 'moda', 
            'estilo', 'fashion', 'coleção', 'tendência', 'look', 'outfit', 'calçado', 'jeans', 'blusa'
        ],
        'Eletrônicos' => [
            'smartphone', 'celular', 'computador', 'notebook', 'tablet', 'tv', 'televisão', 'fone', 
            'câmera', 'gadget', 'tech', 'tecnologia', 'eletrônico', 'console', 'game', 'videogame'
        ],
        'Casa e Decoração' => [
            'móvel', 'decoração', 'casa', 'cozinha', 'quarto', 'sala', 'banheiro', 'jardim', 'sofá', 
            'mesa', 'cadeira', 'iluminação', 'tapete', 'cortina', 'utensílio', 'doméstico', 'lar'
        ],
        'Alimentos e Bebidas' => [
            'comida', 'alimento', 'bebida', 'refeição', 'lanche', 'café', 'chá', 'suco', 'refrigerante', 
            'cerveja', 'vinho', 'destilado', 'orgânico', 'natural', 'gourmet', 'saudável', 'diet', 'light'
        ],
        'Beleza e Cuidados Pessoais' => [
            'maquiagem', 'batom', 'perfume', 'cabelo', 'shampoo', 'condicionador', 'creme', 'hidratante', 
            'protetor solar', 'desodorante', 'sabonete', 'esmalte', 'skincare', 'beleza', 'cosmético'
        ],
        'Esportes' => [
            'esporte', 'treino', 'academia', 'fitness', 'exercício', 'musculação', 'corrida', 'bicicleta', 
            'natação', 'futebol', 'basquete', 'vôlei', 'tênis', 'yoga', 'pilates', 'suplemento'
        ],
        'Brinquedos e Jogos' => [
            'brinquedo', 'jogo', 'boneca', 'carrinho', 'quebra-cabeça', 'tabuleiro', 'lego', 'pelúcia', 
            'educativo', 'infantil', 'diversão', 'criança', 'bebê', 'eletrônico', 'videogame'
        ],
        'Livros e Mídia' => [
            'livro', 'revista', 'jornal', 'quadrinho', 'filme', 'série', 'música', 'cd', 'dvd', 'blu-ray', 
            'streaming', 'digital', 'ebook', 'audiolivro', 'literatura', 'autor', 'leitura'
        ],
        'Saúde e Bem-estar' => [
            'saúde', 'remédio', 'medicamento', 'vitamina', 'suplemento', 'bem-estar', 'farmácia', 'médico', 
            'terapia', 'massagem', 'relaxamento', 'meditação', 'natural', 'orgânico', 'saudável'
        ]
    ];

    /**
     * Categoriza um vídeo com base em seu conteúdo
     *
     * @param array $videoData
     * @return array
     */
    public function categorizeVideo($videoData)
    {
        // Extrair texto do vídeo para análise
        $text = $videoData['ad_creative_body'] . ' ' . $videoData['ad_creative_link_title'];
        $text = strtolower($text);

        // Pontuação para cada categoria
        $categoryScores = [];
        foreach ($this->categoryKeywords as $category => $keywords) {
            $score = 0;
            foreach ($keywords as $keyword) {
                $count = substr_count($text, strtolower($keyword));
                $score += $count;
            }
            $categoryScores[$category] = $score;
        }

        // Determinar a categoria principal
        $mainCategory = 'Outros';
        $maxScore = 0;
        foreach ($categoryScores as $category => $score) {
            if ($score > $maxScore) {
                $maxScore = $score;
                $mainCategory = $category;
            }
        }

        // Se não houver correspondência clara, usar 'Outros'
        if ($maxScore == 0) {
            $mainCategory = 'Outros';
        }

        // Determinar subcategorias
        $subCategories = $this->getSubCategories($text, $mainCategory);

        return [
            'main_category' => $mainCategory,
            'sub_categories' => $subCategories,
            'category_scores' => $categoryScores
        ];
    }

    /**
     * Determina subcategorias com base no texto e categoria principal
     *
     * @param string $text
     * @param string $mainCategory
     * @return array
     */
    protected function getSubCategories($text, $mainCategory)
    {
        $result = [];
        
        // Se a categoria principal não existir na lista de subcategorias, retornar vazio
        if (!isset($this->subCategories[$mainCategory])) {
            return $result;
        }

        // Criar palavras-chave para subcategorias (simplificado)
        $subCategoryKeywords = [];
        foreach ($this->subCategories[$mainCategory] as $subCategory) {
            $keywords = explode(' ', strtolower($subCategory));
            $subCategoryKeywords[$subCategory] = $keywords;
        }

        // Pontuação para cada subcategoria
        $subCategoryScores = [];
        foreach ($subCategoryKeywords as $subCategory => $keywords) {
            $score = 0;
            foreach ($keywords as $keyword) {
                if (strlen($keyword) > 3) { // Ignorar palavras muito curtas
                    $count = substr_count($text, strtolower($keyword));
                    $score += $count;
                }
            }
            $subCategoryScores[$subCategory] = $score;
        }

        // Ordenar por pontuação e pegar as top 3 subcategorias
        arsort($subCategoryScores);
        $topSubCategories = array_slice($subCategoryScores, 0, 3, true);
        
        // Filtrar subcategorias com pontuação zero
        $topSubCategories = array_filter($topSubCategories, function($score) {
            return $score > 0;
        });

        return array_keys($topSubCategories);
    }

    /**
     * Salva um tema no banco de dados
     *
     * @param string $name
     * @param string $description
     * @return Topic
     */
    public function saveTopic($name, $description = '')
    {
        $slug = Str::slug($name);
        
        $topic = Topic::firstOrNew(['slug' => $slug]);
        $topic->name = $name;
        $topic->description = $description;
        $topic->save();
        
        return $topic;
    }

    /**
     * Obtém todas as categorias principais
     *
     * @return array
     */
    public function getMainCategories()
    {
        return $this->retailCategories;
    }

    /**
     * Obtém subcategorias para uma categoria principal
     *
     * @param string $mainCategory
     * @return array
     */
    public function getSubCategoriesForCategory($mainCategory)
    {
        return $this->subCategories[$mainCategory] ?? [];
    }

    /**
     * Analisa temas populares em um conjunto de vídeos
     *
     * @param array $videos
     * @return array
     */
    public function analyzePopularThemes($videos)
    {
        $categoryCount = [];
        $subCategoryCount = [];
        
        foreach ($videos as $video) {
            $category = $video['category'] ?? 'Outros';
            $categoryCount[$category] = ($categoryCount[$category] ?? 0) + 1;
            
            if (isset($video['sub_categories']) && is_array($video['sub_categories'])) {
                foreach ($video['sub_categories'] as $subCategory) {
                    $subCategoryCount[$subCategory] = ($subCategoryCount[$subCategory] ?? 0) + 1;
                }
            }
        }
        
        arsort($categoryCount);
        arsort($subCategoryCount);
        
        return [
            'main_categories' => $categoryCount,
            'sub_categories' => $subCategoryCount
        ];
    }

    /**
     * Analisa o desempenho por categoria
     *
     * @param array $videos
     * @return array
     */
    public function analyzePerformanceByCategory($videos)
    {
        $result = [];
        
        foreach ($this->retailCategories as $category) {
            $categoryVideos = array_filter($videos, function($video) use ($category) {
                return ($video['category'] ?? '') === $category;
            });
            
            if (count($categoryVideos) > 0) {
                $totalViews = array_sum(array_column($categoryVideos, 'views'));
                $totalShares = array_sum(array_column($categoryVideos, 'shares'));
                $totalConversions = array_sum(array_column($categoryVideos, 'conversions'));
                
                $result[$category] = [
                    'videos_count' => count($categoryVideos),
                    'total_views' => $totalViews,
                    'total_shares' => $totalShares,
                    'total_conversions' => $totalConversions,
                    'avg_views' => round($totalViews / count($categoryVideos)),
                    'avg_shares' => round($totalShares / count($categoryVideos)),
                    'avg_conversions' => round($totalConversions / count($categoryVideos)),
                    'conversion_rate' => round($totalConversions / $totalViews * 100, 2)
                ];
            }
        }
        
        return $result;
    }
}
