<?php

namespace App\Http\Controllers;

use App\Services\ThirdPartyVideoAnalyzer;
use Illuminate\Http\Request;

class ThirdPartyAnalysisController extends Controller
{
    /**
     * Serviço de análise de vídeos de terceiros
     *
     * @var ThirdPartyVideoAnalyzer
     */
    protected $thirdPartyVideoAnalyzer;

    /**
     * Construtor
     *
     * @param ThirdPartyVideoAnalyzer $thirdPartyVideoAnalyzer
     */
    public function __construct(ThirdPartyVideoAnalyzer $thirdPartyVideoAnalyzer)
    {
        $this->thirdPartyVideoAnalyzer = $thirdPartyVideoAnalyzer;
    }

    /**
     * Exibe a página inicial de análise de vídeos de terceiros
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        // Obtém as categorias disponíveis para o filtro
        $categories = [
            'Moda',
            'Eletrônicos',
            'Casa e Decoração',
            'Alimentos e Bebidas',
            'Beleza e Cuidados Pessoais',
            'Esportes',
            'Brinquedos e Jogos',
            'Livros e Mídia',
            'Saúde e Bem-estar',
            'Outros'
        ];

        return view('third_party.index', compact('categories'));
    }

    /**
     * Realiza a análise de vídeos de terceiros
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function analyze(Request $request)
    {
        $searchTerm = $request->input('search_term', '');
        $category = $request->input('category', '');

        // Parâmetros de pesquisa
        $searchParams = [
            'search_terms' => $searchTerm,
        ];

        if (!empty($category)) {
            $searchParams['category'] = $category;
        }

        // Realizar análise
        $analysisResults = $this->thirdPartyVideoAnalyzer->analyzeThirdPartyVideos($searchParams);

        // Obtém as categorias disponíveis para o filtro
        $categories = [
            'Moda',
            'Eletrônicos',
            'Casa e Decoração',
            'Alimentos e Bebidas',
            'Beleza e Cuidados Pessoais',
            'Esportes',
            'Brinquedos e Jogos',
            'Livros e Mídia',
            'Saúde e Bem-estar',
            'Outros'
        ];

        return view('third_party.results', [
            'videos' => $analysisResults['videos'],
            'themeAnalysis' => $analysisResults['theme_analysis'],
            'performanceAnalysis' => $analysisResults['performance_analysis'],
            'audienceAnalysis' => $analysisResults['audience_analysis'],
            'categories' => $categories,
            'searchTerm' => $searchTerm,
            'category' => $category,
            'totalResults' => count($analysisResults['videos'])
        ]);
    }

    /**
     * Exibe detalhes da análise de um tema específico
     *
     * @param Request $request
     * @param string $category
     * @return \Illuminate\View\View
     */
    public function themeDetails(Request $request, $category)
    {
        // Parâmetros de pesquisa
        $searchParams = [
            'category' => $category,
        ];

        // Realizar análise
        $analysisResults = $this->thirdPartyVideoAnalyzer->analyzeThirdPartyVideos($searchParams);

        // Filtrar apenas os vídeos da categoria selecionada
        $videos = array_filter($analysisResults['videos'], function($video) use ($category) {
            return $video['category'] === $category;
        });

        return view('third_party.theme_details', [
            'videos' => array_values($videos),
            'category' => $category,
            'performanceAnalysis' => $analysisResults['performance_analysis'][$category] ?? [],
            'audienceAnalysis' => $analysisResults['audience_analysis'][$category] ?? [],
            'totalResults' => count($videos)
        ]);
    }

    /**
     * Exibe análise comparativa de temas
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function compareThemes(Request $request)
    {
        $categories = $request->input('categories', []);

        if (empty($categories) || count($categories) < 2) {
            return redirect()->route('third_party.index')
                ->with('error', 'Selecione pelo menos duas categorias para comparar.');
        }

        // Realizar análise para cada categoria
        $results = [];
        foreach ($categories as $category) {
            $searchParams = [
                'category' => $category,
            ];
            $analysisResults = $this->thirdPartyVideoAnalyzer->analyzeThirdPartyVideos($searchParams);
            
            $results[$category] = [
                'performance' => $analysisResults['performance_analysis'][$category] ?? [],
                'audience' => $analysisResults['audience_analysis'][$category] ?? [],
                'videos_count' => count(array_filter($analysisResults['videos'], function($video) use ($category) {
                    return $video['category'] === $category;
                }))
            ];
        }

        return view('third_party.compare_themes', [
            'results' => $results,
            'categories' => $categories
        ]);
    }

    /**
     * Exibe o dashboard de análise de tendências
     *
     * @return \Illuminate\View\View
     */
    public function trendsDashboard()
    {
        // Realizar análise geral
        $analysisResults = $this->thirdPartyVideoAnalyzer->analyzeThirdPartyVideos([]);

        // Ordenar categorias por desempenho
        $categoryPerformance = [];
        foreach ($analysisResults['performance_analysis'] as $category => $data) {
            $categoryPerformance[$category] = $data['total_conversions'];
        }
        arsort($categoryPerformance);

        // Obter os vídeos mais populares
        $topVideos = $analysisResults['videos'];
        usort($topVideos, function($a, $b) {
            return $b['conversions'] <=> $a['conversions'];
        });
        $topVideos = array_slice($topVideos, 0, 5);

        return view('third_party.trends_dashboard', [
            'themeAnalysis' => $analysisResults['theme_analysis'],
            'performanceAnalysis' => $analysisResults['performance_analysis'],
            'audienceAnalysis' => $analysisResults['audience_analysis'],
            'categoryPerformance' => $categoryPerformance,
            'topVideos' => $topVideos
        ]);
    }
}
