<?php

namespace App\Http\Controllers;

use App\Models\Page;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;

class PageController extends Controller
{
    /**
     * Exibe a lista de páginas do usuário.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $pages = Page::where('user_id', Auth::id())->get();
        return view('pages.index', compact('pages'));
    }

    /**
     * Importa as páginas do usuário do Facebook.
     *
     * @return \Illuminate\Http\Response
     */
    public function import()
    {
        try {
            $user = Auth::user();
            
            // Verificar se o token do Facebook está disponível
            if (empty($user->facebook_token)) {
                return redirect()->route('dashboard')
                    ->with('error', 'Token do Facebook não encontrado. Por favor, faça login novamente.');
            }
            
            // Fazer requisição à API do Facebook para obter as páginas
            $response = Http::get('https://graph.facebook.com/v18.0/me/accounts', [
                'access_token' => $user->facebook_token,
                'fields' => 'id,name,access_token,category,fan_count,picture'
            ]);
            
            if (!$response->successful()) {
                return redirect()->route('dashboard')
                    ->with('error', 'Erro ao importar páginas: ' . $response->json('error.message', 'Erro desconhecido'));
            }
            
            $pagesData = $response->json('data', []);
            $importCount = 0;
            
            foreach ($pagesData as $pageData) {
                // Atualizar ou criar a página no banco de dados
                $page = Page::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'facebook_page_id' => $pageData['id']
                    ],
                    [
                        'name' => $pageData['name'],
                        'access_token' => $pageData['access_token'],
                        'token_expires_at' => Carbon::now()->addDays(60), // Tokens de página geralmente não expiram, mas definimos 60 dias por segurança
                        'category' => $pageData['category'] ?? null,
                        'follower_count' => $pageData['fan_count'] ?? 0,
                        'profile_picture_url' => $pageData['picture']['data']['url'] ?? null,
                    ]
                );
                
                $importCount++;
            }
            
            return redirect()->route('pages.index')
                ->with('success', "Importação concluída! {$importCount} páginas foram importadas com sucesso.");
            
        } catch (\Exception $e) {
            return redirect()->route('dashboard')
                ->with('error', 'Erro ao importar páginas: ' . $e->getMessage());
        }
    }

    /**
     * Exibe os detalhes de uma página específica.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $page = Page::where('user_id', Auth::id())->findOrFail($id);
        $videos = $page->videos()->orderBy('published_at', 'desc')->paginate(10);
        
        return view('pages.show', compact('page', 'videos'));
    }

    /**
     * Remove uma página do sistema.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $page = Page::where('user_id', Auth::id())->findOrFail($id);
        $pageName = $page->name;
        
        // A exclusão em cascata dos vídeos e métricas relacionados é tratada pelas migrações
        $page->delete();
        
        return redirect()->route('pages.index')
            ->with('success', "A página '{$pageName}' foi removida com sucesso.");
    }
}
