<?php

namespace App\Http\Controllers;

use App\Models\Page;
use App\Models\Video;
use App\Models\VideoMetric;
use App\Models\AudienceSegment;
use App\Models\Topic;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MetricsController extends Controller
{
    /**
     * Exibe o dashboard de métricas.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        
        // Obter páginas do usuário
        $pages = Page::where('user_id', $user->id)->get();
        $pageIds = $pages->pluck('id')->toArray();
        
        // Obter estatísticas gerais
        $totalVideos = Video::whereIn('page_id', $pageIds)->count();
        $totalViews = 0;
        $totalEngagement = 0;
        
        // Obter vídeos mais populares
        $popularVideos = Video::whereIn('page_id', $pageIds)
            ->join('video_metrics', 'videos.id', '=', 'video_metrics.video_id')
            ->where('video_metrics.metric_name', 'total_video_views')
            ->select('videos.*', 'video_metrics.metric_value as views')
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();
        
        // Calcular visualizações totais
        $viewsMetrics = VideoMetric::whereIn('metric_name', ['total_video_views', 'views'])
            ->whereHas('video', function($query) use ($pageIds) {
                $query->whereIn('page_id', $pageIds);
            })
            ->get();
            
        foreach ($viewsMetrics as $metric) {
            $totalViews += (int)$metric->metric_value;
        }
        
        // Calcular engajamento total (reações, comentários, compartilhamentos)
        $engagementMetrics = VideoMetric::whereIn('metric_name', [
                'total_video_reactions_by_type_total',
                'total_video_comments',
                'total_video_shares'
            ])
            ->whereHas('video', function($query) use ($pageIds) {
                $query->whereIn('page_id', $pageIds);
            })
            ->get();
            
        foreach ($engagementMetrics as $metric) {
            if ($metric->metric_name == 'total_video_reactions_by_type_total') {
                $reactions = json_decode($metric->metric_value, true);
                if (is_array($reactions)) {
                    $totalEngagement += array_sum($reactions);
                }
            } else {
                $totalEngagement += (int)$metric->metric_value;
            }
        }
        
        // Obter dados de tendências (últimos 30 dias)
        $startDate = Carbon::now()->subDays(30);
        $endDate = Carbon::now();
        
        $trendData = Video::whereIn('page_id', $pageIds)
            ->where('published_at', '>=', $startDate)
            ->orderBy('published_at')
            ->get()
            ->groupBy(function($video) {
                return $video->published_at->format('Y-m-d');
            })
            ->map(function($videos) {
                return [
                    'count' => $videos->count(),
                    'views' => $this->getViewsForVideos($videos)
                ];
            });
            
        // Obter dados demográficos agregados
        $demographicData = AudienceSegment::whereHas('video', function($query) use ($pageIds) {
                $query->whereIn('page_id', $pageIds);
            })
            ->where('segment_type', 'demographic')
            ->select('segment_value', DB::raw('SUM(view_count) as total_views'), DB::raw('AVG(engagement_rate) as avg_engagement'))
            ->groupBy('segment_value')
            ->orderBy('total_views', 'desc')
            ->get();
            
        // Obter dados geográficos agregados
        $locationData = AudienceSegment::whereHas('video', function($query) use ($pageIds) {
                $query->whereIn('page_id', $pageIds);
            })
            ->where('segment_type', 'location')
            ->select('segment_value', DB::raw('SUM(view_count) as total_views'), DB::raw('AVG(engagement_rate) as avg_engagement'))
            ->groupBy('segment_value')
            ->orderBy('total_views', 'desc')
            ->limit(10)
            ->get();
            
        return view('metrics.index', compact(
            'pages', 
            'totalVideos', 
            'totalViews', 
            'totalEngagement', 
            'popularVideos', 
            'trendData', 
            'demographicData', 
            'locationData'
        ));
    }
    
    /**
     * Exibe a análise comparativa de vídeos.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function compare(Request $request)
    {
        $user = Auth::user();
        $pages = Page::where('user_id', $user->id)->get();
        $pageIds = $pages->pluck('id')->toArray();
        
        // Obter vídeos para comparação
        $videoIds = $request->input('video_ids', []);
        $videos = [];
        $metrics = [];
        $audienceData = [];
        
        if (!empty($videoIds)) {
            $videos = Video::whereIn('id', $videoIds)
                ->whereIn('page_id', $pageIds)
                ->get();
                
            foreach ($videos as $video) {
                // Obter métricas para cada vídeo
                $videoMetrics = $video->metrics()
                    ->whereIn('metric_name', [
                        'total_video_views',
                        'total_video_views_unique',
                        'total_video_complete_views',
                        'total_video_30s_views',
                        'total_video_avg_time_watched'
                    ])
                    ->get()
                    ->keyBy('metric_name');
                    
                $metrics[$video->id] = $videoMetrics;
                
                // Obter dados de público para cada vídeo
                $audienceData[$video->id] = [
                    'demographic' => $video->audienceSegments()
                        ->where('segment_type', 'demographic')
                        ->orderBy('view_count', 'desc')
                        ->get(),
                    'location' => $video->audienceSegments()
                        ->where('segment_type', 'location')
                        ->orderBy('view_count', 'desc')
                        ->limit(5)
                        ->get()
                ];
            }
        }
        
        // Obter todos os vídeos para seleção
        $allVideos = Video::whereIn('page_id', $pageIds)
            ->orderBy('published_at', 'desc')
            ->get();
            
        return view('metrics.compare', compact('pages', 'videos', 'allVideos', 'metrics', 'audienceData'));
    }
    
    /**
     * Exibe a análise de público-alvo.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function audience(Request $request)
    {
        $user = Auth::user();
        $pages = Page::where('user_id', $user->id)->get();
        $pageIds = $pages->pluck('id')->toArray();
        
        // Filtros
        $pageId = $request->input('page_id');
        $dateFrom = $request->input('date_from') ? Carbon::parse($request->input('date_from')) : null;
        $dateTo = $request->input('date_to') ? Carbon::parse($request->input('date_to')) : null;
        
        // Construir query base
        $query = AudienceSegment::whereHas('video', function($q) use ($pageIds, $pageId, $dateFrom, $dateTo) {
            $q->whereIn('page_id', $pageId ? [$pageId] : $pageIds);
            
            if ($dateFrom) {
                $q->where('published_at', '>=', $dateFrom);
            }
            
            if ($dateTo) {
                $q->where('published_at', '<=', $dateTo);
            }
        });
        
        // Dados demográficos
        $demographicData = (clone $query)
            ->where('segment_type', 'demographic')
            ->select('segment_value', DB::raw('SUM(view_count) as total_views'), DB::raw('AVG(engagement_rate) as avg_engagement'))
            ->groupBy('segment_value')
            ->orderBy('total_views', 'desc')
            ->get();
            
        // Processar dados demográficos para gráficos
        $ageGenderData = [];
        foreach ($demographicData as $item) {
            $parts = explode('_', $item->segment_value);
            if (count($parts) == 2) {
                $gender = $parts[0];
                $ageGroup = $parts[1];
                
                if (!isset($ageGenderData[$gender])) {
                    $ageGenderData[$gender] = [];
                }
                
                $ageGenderData[$gender][$ageGroup] = [
                    'views' => $item->total_views,
                    'engagement' => $item->avg_engagement
                ];
            }
        }
        
        // Dados geográficos
        $locationData = (clone $query)
            ->where('segment_type', 'location')
            ->select('segment_value', DB::raw('SUM(view_count) as total_views'), DB::raw('AVG(engagement_rate) as avg_engagement'))
            ->groupBy('segment_value')
            ->orderBy('total_views', 'desc')
            ->limit(10)
            ->get();
            
        // Obter vídeos mais populares por segmento demográfico
        $popularByDemographic = [];
        $demographicSegments = ['F_18-24', 'M_18-24', 'F_25-34', 'M_25-34', 'F_35-44', 'M_35-44'];
        
        foreach ($demographicSegments as $segment) {
            $topVideos = Video::whereIn('page_id', $pageId ? [$pageId] : $pageIds)
                ->whereHas('audienceSegments', function($q) use ($segment) {
                    $q->where('segment_type', 'demographic')
                        ->where('segment_value', $segment);
                })
                ->join('audience_segments', 'videos.id', '=', 'audience_segments.video_id')
                ->where('audience_segments.segment_type', 'demographic')
                ->where('audience_segments.segment_value', $segment)
                ->select('videos.*', 'audience_segments.view_count', 'audience_segments.engagement_rate')
                ->orderBy('audience_segments.view_count', 'desc')
                ->limit(3)
                ->get();
                
            if ($topVideos->isNotEmpty()) {
                $popularByDemographic[$segment] = $topVideos;
            }
        }
        
        return view('metrics.audience', compact(
            'pages', 
            'pageId', 
            'dateFrom', 
            'dateTo', 
            'demographicData', 
            'ageGenderData', 
            'locationData', 
            'popularByDemographic'
        ));
    }
    
    /**
     * Exibe a análise de temas.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function topics(Request $request)
    {
        $user = Auth::user();
        $pages = Page::where('user_id', $user->id)->get();
        $pageIds = $pages->pluck('id')->toArray();
        
        // Filtros
        $pageId = $request->input('page_id');
        $topicId = $request->input('topic_id');
        
        // Obter todos os temas
        $topics = Topic::whereHas('videos', function($q) use ($pageIds, $pageId) {
                $q->whereIn('page_id', $pageId ? [$pageId] : $pageIds);
            })
            ->withCount(['videos' => function($q) use ($pageIds, $pageId) {
                $q->whereIn('page_id', $pageId ? [$pageId] : $pageIds);
            }])
            ->orderBy('videos_count', 'desc')
            ->get();
            
        // Análise de tema específico
        $selectedTopic = null;
        $topicVideos = collect();
        $topicPerformance = [];
        
        if ($topicId) {
            $selectedTopic = Topic::find($topicId);
            
            if ($selectedTopic) {
                $topicVideos = $selectedTopic->videos()
                    ->whereIn('page_id', $pageId ? [$pageId] : $pageIds)
                    ->withPivot('relevance_score')
                    ->orderBy('pivot_relevance_score', 'desc')
                    ->get();
                    
                // Calcular performance do tema ao longo do tempo
                $videoIds = $topicVideos->pluck('id')->toArray();
                $viewsData = VideoMetric::whereIn('video_id', $videoIds)
                    ->whereIn('metric_name', ['total_video_views', 'views'])
                    ->join('videos', 'video_metrics.video_id', '=', 'videos.id')
                    ->select('videos.published_at', 'video_metrics.metric_value')
                    ->orderBy('videos.published_at')
                    ->get()
                    ->groupBy(function($item) {
                        return Carbon::parse($item->published_at)->format('Y-m-d');
                    })
                    ->map(function($items) {
                        $total = 0;
                        foreach ($items as $item) {
                            $total += (int)$item->metric_value;
                        }
                        return $total;
                    });
                    
                $topicPerformance = $viewsData->toArray();
            }
        }
        
        // Obter temas mais populares (por visualizações)
        $popularTopics = Topic::whereHas('videos', function($q) use ($pageIds, $pageId) {
                $q->whereIn('page_id', $pageId ? [$pageId] : $pageIds);
            })
            ->withCount(['videos' => function($q) use ($pageIds, $pageId) {
                $q->whereIn('page_id', $pageId ? [$pageId] : $pageIds);
            }])
            ->orderBy('videos_count', 'desc')
            ->limit(5)
            ->get();
            
        return view('metrics.topics', compact(
            'pages', 
            'pageId', 
            'topics', 
            'selectedTopic', 
            'topicVideos', 
            'topicPerformance', 
            'popularTopics'
        ));
    }
    
    /**
     * Obtém o total de visualizações para uma coleção de vídeos.
     *
     * @param  \Illuminate\Support\Collection  $videos
     * @return int
     */
    private function getViewsForVideos($videos)
    {
        $totalViews = 0;
        $videoIds = $videos->pluck('id')->toArray();
        
        $viewsMetrics = VideoMetric::whereIn('video_id', $videoIds)
            ->whereIn('metric_name', ['total_video_views', 'views'])
            ->get();
            
        foreach ($viewsMetrics as $metric) {
            $totalViews += (int)$metric->metric_value;
        }
        
        return $totalViews;
    }
}
